/* 
Multi Code Lock with 10 Access Codes
Written in Atmel Studio 6
AVR C program for ATMEGA8A
Author: Mahmood Alimohammadi
Date: November 14,2019
***************************************************************
Description: This is a multi code lock with a 10-digit master
code feature to increase security. It is capable of storing up 
to 10 access codes using 10 ID codes for the users to allow the 
access codes to be entered or changed.The input device is a 4X3 
matrix keypad. The lock has a 16X2 LCD module as a status display. 
Moreover, it is equipped with two automated re-locking features 
ensuring you never leave your door unlocked. It also uses a 
countdown counter limiting the unlocked state time needed for 
entering the ID code or a new access code. There are two options: 
1. Using the 10 digit master code+ 4-digit ID codes for entering 
the access codes. 2. Only using the ID codes first to enter or 
change the access codes.  
***************************************************************
*/
#define F_CPU 1000000UL

#include <avr/io.h>
#include <stdio.h>
#include <util/delay.h>
#include <avr/eeprom.h>

#include "lcd.h"

#define LED1 PC1
#define SOLENOID PC2
#define LED2 PC3
#define jumper PC5

//EEPROM address 
uint16_t EEMEM ee_code[10];

//ID codes for 10 users
const uint16_t ID[10]={1001,1002,1003,1004,1005,1006,1007,1008,1009,1010};
// 10 digit master code
const unsigned long int mastercode=1234567890;

//Global Variables Declarations
uint8_t digit, pressedKey, access, count , index ;
unsigned long int value;
unsigned long int entrycode, key;
uint16_t code[10];

/*-------------------------------------------------------------*/
/*Function Declarations*/
unsigned long int Getkeypad();
void accessgranted();
uint8_t idindex();
uint8_t codeindex();
void accessdenied();
void newcodesaved();

/***************************************************************/
void  main()
{
	uint8_t i=0,c=30, countdown[10], master=0;
	uint16_t masterkey=0;


	begin:
	
	//LCD initialize.
	lcd_init(LCD_DISP_ON);
	lcd_clrscr();
	
	entrycode=0;
	value=0;
	key=0;
	digit=0;
	c=30;
	
	DDRC=0xDF;
	PORTC=0x20;	
	
	lcd_clrscr();
	lcd_gotoxy(3,0);
	lcd_puts("Enter Code");
	_delay_ms(100);
	
	/*Read the codes from the EEPROM.*/
	code[0] = eeprom_read_word(&ee_code[0]);
	code[1] = eeprom_read_word(&ee_code[1]);
	code[2] = eeprom_read_word(&ee_code[2]);
	code[3] = eeprom_read_word(&ee_code[3]);
	code[4] = eeprom_read_word(&ee_code[4]);

	code[5] = eeprom_read_word(&ee_code[5]);
	code[6] = eeprom_read_word(&ee_code[6]);
	code[7] = eeprom_read_word(&ee_code[7]);
	code[8] = eeprom_read_word(&ee_code[8]);
	code[9] = eeprom_read_word(&ee_code[9]);
	/***************************************************************/
	while(1)
	{
	
		/*Call the Getkeypad function.*/
		key=Getkeypad();
		
		//LED1 OFF
		PORTC &= ~(1<<LED1);
		
		//SOLENOID OFF
		PORTC &= ~(1<<SOLENOID);
			
		/* If jumper(JP1) is not connected to PC5 and ground...*/
		if bit_is_set(PINC,jumper)
		{
		
			if (digit==10 && master==0)
				entrycode=key;
			 
			/* If master code is entered and # is pressed...*/	 		
			if(digit==11 && pressedKey==12)
			{
				key=0;
				if (entrycode==mastercode)
				{
					master=1;
					accessgranted();
					goto begin;
				}
					
			 }
		 }				
/*-------------------------------------------------------------*/
		if (digit==4 )
			entrycode=key;
			
		/* If Id code or access code is entered and # is pressed...*/
		if (digit==5 && pressedKey==12 )
		{
			key=0;
			
			/*If jumper is connected to PC5 and ground...*/
			if bit_is_clear(PINC,jumper)
				master=1;
								
			/* If entry code=ID code and master=1...*/
			if (entrycode==ID[idindex()] && master==1 )
			{
				masterkey=ID[index];
				access=1;
			}
								
			else
			{	
			/* If entry code=access code...*/
				if(entrycode==code[codeindex()])	
				masterkey=code[index];
			}		
/*-------------------------------------------------------------*/		
								
			if (entrycode==masterkey )
			{
				accessgranted();
				masterkey=0;
				goto begin;				
			}
						
			else
			{
				accessdenied();							
				goto begin;
			}
			
		}
/*-------------------------------------------------------------*/
		/*If digit=5 and # or * is not pressed,clear the entry code.*/
		
		while (digit==5 && pressedKey !=10 && pressedKey !=12 && master==1)					
			goto begin;	
					
		//clear the entry code.
		while (digit>10 && pressedKey !=12 && master==0)
			goto begin;	
		
		//clear the entry code.
		while(digit>5 && digit<11 && pressedKey==12)
			goto begin;
			
		//clear the entry code.	
		while (digit>5 && bit_is_clear(PINC,jumper))
			goto begin;
			
		//clear the entry code.
		while (access==0 && pressedKey==10 && master==0)
			goto begin;		
	
		//clear the entry code.
		while (digit<5 && pressedKey==10 && master==1 )	
			goto begin;				
		
		//clear the entry code.
		while (access==1 && digit<5 && pressedKey==10)		
			goto begin;
		
/*-------------------------------------------------------------*/
		/* If access is granted(access=1) , and the digits
		 of the new code=4 + *(10)...*/
		while (access==1 && digit==5 && pressedKey==10 )
		{			
			key=0;
			
			//Write the defined entry code to the EEPROM.
			code[index]= entrycode;
			
			/* If master code is correct 
			or Jumper is connected to PC5 and ground...*/
			if (master==1 )
			
				/* Write the code to the EEPROM*/
				eeprom_write_word(&ee_code[index], code[index]); 						
			
			_delay_ms(50);
			newcodesaved();	
			master=0;
			goto begin;
						
		}
				
		//If access is granted or master code is entered...
		if (access==1 || master==1 && access==0)
		{
			if(access==1)
			{
				lcd_gotoxy(0,0);
				lcd_puts("Enter New Code");
			}
			
			else if(master==1 )
			
			{
				//If jumper(JP1) is connected to PC5 and ground...
				if(bit_is_clear(PINC,jumper))
				{
					access=0;
					master=0;
					goto begin;
				}				
				
				lcd_gotoxy(0,0);
				lcd_puts("Enter ID Code");
			}
			
			itoa(c,countdown,10);
			lcd_gotoxy(0,1);

			lcd_puts(countdown);
			
			if(c<10)
			{
				lcd_gotoxy(1,1);
				lcd_puts(" ");
			}
				
			c--;
			_delay_ms(500);	
				
			//Counting down from 30 to 0 for entering the  ID code or new code.
			if (c<1)
			{
				access=0;
				master=0;
				goto begin;
			}
			
		}
		
	}
	
}
/***************************************************************/
//GetKeypad function
unsigned long int Getkeypad()
{
	uint8_t row=0, col=0 ,d=0 ;
	
	/*PORTB is used for the Keypad.
	Row1=PB3, Row2=PB2, Row3=PB1, Row4=PB0
	Column1=PB7, Column2=PB6, Column3=PB5*/
	
	PORTB|=0X0F;
	
	for(col=0;col<3;col++)
	{
		DDRB&=~(0X7F);
		switch(col)
		{
			case 0:	DDRB = (0X80); break;
			case 1:	DDRB = (0X40); break;
			case 2:	DDRB = (0X20); break;
		}
		
		for(row=0;row<4;row++)
		{
			if(!(PINB & (0X08>>row)))
			{
				//LED on PORTC.1=ON(key press confirm)
				PORTC |=(1<<LED1);
				
				uint8_t keynumber=(row*3+col);
				
				//keypad keys= (1-9 keys + 10=*,11=0, and 12=#)		
				const uint8_t lookup[12]={1,2,3,4,5,6,7,8,9,10,0,12};	
				pressedKey= lookup[keynumber];
				
				/*Combine multiple numbers into one number by 
				multiplying the previous value by 10 and adding 
				this new value.*/
				value=value*10;
				value=value+pressedKey;
				
				digit = digit+1;
				d = digit + 4;
				
				//Put ***** for the digits of the entry code.
				lcd_gotoxy(d,1);
				lcd_puts("*");
				
				//_delay_ms(100);
				_delay_ms(500);
				
			}
			
		}
		
	}
	return value;
}
/*-------------------------------------------------------------*/
void accessgranted()
{
	uint8_t i=0;
	if(access==0)
		//Solenoid on PORTC.2=ON
		PORTC |=(1<<SOLENOID);
	lcd_clrscr();
	lcd_gotoxy(1,0);
	lcd_puts("Access Granted");
	count=0;
	
	/* progress bar progresses to the right in 5 seconds.*/
	for(i=0;i<16;i++)
	{
		lcd_gotoxy(i,1);
		lcd_puts("*");
		_delay_ms(150);
		
		//LED2 on PC3 blinks.
		PORTC |= (1 << LED2);
		_delay_ms(150);
		PORTC &= ~(1<<LED2);
	}
	
}
/*-------------------------------------------------------------*/
uint8_t idindex()
{
	/* If entry code=ID0-ID9=1001-1009 ID codes...*/
	 if (entrycode==ID[0])
		index=0;

	else if (entrycode==ID[1])
		index=1;

	else if (entrycode==ID[2])
		index=2;

	else if (entrycode==ID[3])
		index=3;

	else if (entrycode==ID[4])
		index=4;

	else if (entrycode==ID[5])
		index=5;

	else if (entrycode==ID[6])
		index=6;

	else if (entrycode==ID[7])
		index=7;

	else if (entrycode==ID[8])
		index=8;

	else if (entrycode==ID[9])
		index=9;

	/*Return the index number of the ID code.*/
	return index;
}

/*-------------------------------------------------------------*/
void accessdenied()
{ 
	uint8_t i=0 ;
	lcd_clrscr();
	lcd_gotoxy(2,0);
	lcd_puts("Access Denied");
	access=0;
	
	/* The progress bar progresses to the right in 1.5 seconds.*/
	for(i=0;i<16;i++)
	{
		lcd_gotoxy(i,1);
		lcd_puts("*");
		_delay_ms(100);
	}
	
	 count++;

	/* Three successive entries of the invalid codes*/
	if (count==3)
	{
		lcd_clrscr();
		lcd_gotoxy(3,0);
		lcd_puts("Try Later");
		
		/*Progression of the Progress bar in 60 seconds*/
		for(i=0;i<16;i++)
		{
			lcd_gotoxy(i,1);
			lcd_puts("*");
			_delay_ms(4000);
		}

		count=0;
	
	}
}	
	
/*-------------------------------------------------------------*/
void newcodesaved()
{
	uint8_t i=0;
	lcd_clrscr();
	lcd_gotoxy(1,0);
	lcd_puts("New Code Saved");
	
	/*Progression of the Progress bar in 3 seconds*/
	for(i=0;i<16;i++)
	{
		lcd_gotoxy(i,1);
		lcd_puts("*");
		access=0;
		
		_delay_ms(200);
	}
	
}

/*-------------------------------------------------------------*/


uint8_t codeindex()
{
	
	// If entry code=code0-code9...
	if(entrycode==code[0])
		index=0;

	else if(entrycode==code[1])
		index=1;

	else if(entrycode==code[2])
		index=2;

	else if(entrycode==code[3])
		index=3;

	else if(entrycode==code[4])
		index=4;

	else if(entrycode==code[5])
		index=5;
	
	else if(entrycode==code[6])
		index=6;
	
	else if(entrycode==code[7])
		index=7;
	
	else if(entrycode==code[8])
		index=8;
	
	else if(entrycode==code[9])
		index=9;
	
	/* Return the index number of the code.*/
	return index;				
}		
/***************************************************************/
//End of Program




